/**
 * Copyright (c), Klass&Ihlenfeld Verlag GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    * Neither the name of the Klass&Ihlenfeld Verlag GmbH nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Alexander Merz, am@golem.de
 */
 
var ws = require("nodejs-websocket")
	, connect = require('connect')
	, noble = require('noble');

var UUID_DEVICE = '7c5203ca4a19430184b039b2e40d11a0' 
	, UUID_SERVICE = '7c5203ca4a19430184b039b2e40d11a1'
	, UUID_CHARACTERISTIC_CMD = '7c5203ca4a19430184b039b2e40d11a2'
	, UUID_CHARACTERISTIC_VOL = '7c5203ca4a19430184b039b2e40d11a3'	
	, UUID_CHARACTERISTIC_TRK = '7c5203ca4a19430184b039b2e40d11a4'		
	, UUID_DESCRIPTION = '7c5203ca4a19430184b039b2e40d11a5';

// Maps the player commands of the websocket to the bluetooth commands
var command_map = 
	{ 'play' : 1
		, 'stop' : 2
		, 'nexttrack' : 3
		, 'prevtrack' : 4
		, 'mute' : 5
		}

// references the bluetooth characteristics
var cmd_char = null
	, cmd_char = null
	, trk_char = null
	, vol_char = null;

// Contains command received while the bluetooth is not available
var cmd_buffer = [];
	
var ws_conn	= [] 	// the websocket connections
	, clean_up = [] // dropped websocket connections
    , message = {success:false, cmd: '', data: '', volume:0, track:''}; // basic reply to client
    
// the callback is called for each new websocket connection    
var server = ws.createServer(function (conn) {

	ws_conn.push(conn);

    console.log("New connection");
    
    var data = ''
    	, cmd = ''
    	, vol = 0
    	, cmd_data = {cmd:'', vol:''};
    
    conn.on("text", function (str) {

		console.log("Received: "+str);
		
		try {
			var cmd_data = JSON.parse(str);
		} catch(e) {
			console.log('No json');
			message.success = false;
			conn.sendText(JSON.stringify(message));
			return;
		}
		
		// send current state on 'nop' command
		if(cmd_data.cmd && 'nop' == cmd_data.cmd) {		
			message.success = true;
			message.cmd = 'nop';
			conn.sendText(JSON.stringify(message));
	    	return;		
		}

		// change volume via bluetooth if volume setting is present
		if(cmd_data.vol && '' != cmd_data.vol) {
			var vol = parseInt(cmd_data.vol);
			console.log("Set Volume to "+vol);								
	    	sendVolume(vol);	
	    	return;
		}
		
		// try to send a command via bluetooth
		if(cmd_data.cmd && '' != cmd_data.cmd) {
			    		
			var cmd = null;
			
			for(var k in command_map) {
				if(cmd_data.cmd == k) {
					cmd = command_map[k];
					break;
				}
			}
						
			if(null != cmd)	{
    			console.log("Command: "+cmd+"");	    		
    			sendCommand(cmd);	    				
				message.success = true;
				message.cmd = cmd_data.cmd;
				conn.sendText(JSON.stringify(message));

			} else {
    			console.log("Unknown command: "+cmd_data.cmd);	    		
				message.success = false;
				message.cmd = cmd_data.cmd;
				conn.sendText(JSON.stringify(message));			
			}		
		    
		    return;    					
		}

    });

    conn.on("close", function (code, reason) {
        console.log("Connection closed")
    })
}).listen(8001); // websocket listens to port 8001

// also create a regular http server on port 8000 to access 
// the html file through the network
connect().use(connect.static(__dirname)).listen(8000);

// Send a command via bluetooth low energy
function sendCommand(cmd) {

	if(null != cmd_char) {

		var buffer = new Buffer(1);	
		buffer.writeUInt8(cmd, 0);
		try {
			cmd_char.write(buffer, false);
		} catch(e) {
			cmd_buffer.push({type:'cmd', cmd:cmd});
			console.log("ERROR CMD");
		}
	
	}

}

// Set the volume via bluetooth low energy
function sendVolume(vol) {
	console.log('VOL to send: '+vol);
	
	if(null != vol_char) {

		var buffer = new Buffer(1);	
		buffer.writeUInt8(vol, 0);
		
		try {
			vol_char.write(buffer, false);
		} catch(e) {
			console.log("ERROR VOL");
		}
	
	}

}

// Our device was discovered
function discover(peripherial) {

	console.log("Device discovered");
	
	peripherial.connect(function(error) {	
		if(error) {
			console.log(error);	
			return;
		}
		
		console.log("=== CONNECT ===");
		
		peripherial.on('disconnect', function() {noble.startScanning([UUID_DEVICE], false);	});		
		
		peripherial.discoverServices([UUID_SERVICE], function(error, services) {		
			if(error) {
				console.log(error);
				return;
			}
			
			console.log('Services discovered');			
			
			for(var i = 0, l = services.length; i < l; i++) {
			
				var service = services[i];
				
				if(UUID_SERVICE == service.uuid)	{
					noble.stopScanning();
					console.log("Specific service found");
					handleService(service);
				}		 
			}
								
		});
	});
};

function handleService(service) {

	service.discoverCharacteristics([], function(error, characteristics) {
		for(var i = 0, l = characteristics.length; i < l; i++) {
		
			switch(characteristics[i].uuid) {
				case UUID_CHARACTERISTIC_CMD :
					console.log('CMD characteristic found');
					cmd_char = characteristics[i];
					break;		
								
				case UUID_CHARACTERISTIC_VOL :
					console.log('VOL characteristic found');
					vol_char = characteristics[i];
					vol_char.notify(true);					
					vol_char.on('read', function(value, isNotification) {
					
						// message from the bluetooth device about a 
						// volume setting
					
						console.log('Volume: '+value.readUInt8(0));
						
						message.success = true;
						message.cmd = '';
						message.volume = value.readUInt8(0);
							
						if(ws_conn.length) {												
							for(var i = 0, l = ws_conn.length; i < l; i++) {
								try { // send the value to all websocket clients
									ws_conn[i].sendText(JSON.stringify(message));
								} catch(e) {
									clean_up.push(i);
								}
							}
						}
						
						cleanUpWsConnections();												
												
					});
					
					break;	

				case UUID_CHARACTERISTIC_TRK :
					console.log('TRK characteristic found');
					trk_char = characteristics[i];
					trk_char.notify(true);
					trk_char.on('read', function(value, isNotification) {
					
						// message from the bluetooth device about 
						// the current track
					
						console.log(value.toString());
						
						message.success = true;
						message.cmd = '';
						message.track = value.toString();
						
						if(ws_conn.length) {
							for(var i = 0, l = ws_conn.length; i < l; i++) {						
								try { // send the track to all websocket clients
									ws_conn[i].sendText(JSON.stringify(message));
								} catch(e) {
									clean_up.push(i);
								}
							}
						}
						
						cleanUpWsConnections();																		
					});
					break;	

			}
		
		}

		if(0 < cmd_buffer.length) {
			var old_buffer = cmd_buffer;
			
			cmd_buffer = [];
															
			for(var i = 0, l = old_buffer.length; i < l; i++) {
				var b = old_buffer[i];
				if('cmd' == b.type) {
					sendCommand(b.cmd);
				} 
			}
			
			console.log(old_buffer);
		}
		
	});
}

/**
 * Remove websocket connections that are not connected anymore
 */
function cleanUpWsConnections() {
	if(clean_up.length) {
		var newws = [];
		
		for(var i = 0, l = ws_conn.length; i < l; i++) {
		
			var found = false;
			
			for(var j = 0, k = clean_up.length; j < k; j++) {
				if(clean_up[k] == i) {
					found = true;
					break;
				}
			}
		
			if(!found) {
				newws.push(ws_conn[i]);
			}
		}
				
		ws_conn = newws;
	}
}

// Start the bluetooth functionality
noble.on('discover', discover);
noble.on('disconnect', function() {
	console.log('Try to reconnect');
	noble.startScanning([UUID_DEVICE], false); // Try to reconnect, if connection broke
});

// look for our device only
noble.startScanning([UUID_DEVICE], false);
console.log("Scanning started");

