/**
 * Copyright (c), Klass&Ihlenfeld Verlag GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    * Neither the name of the Klass&Ihlenfeld Verlag GmbH nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Alexander Merz, am@golem.de
 */
 
 /**
  * This script turns a mac into a bluetooth low energy peripheral
  * and controls an itunes player
  */
 
var sys = require('sys')
	, exec = require('child_process').exec
	, bleno = require('bleno')
	, util = require('util');

var UUID_DEVICE = '7c5203ca4a19430184b039b2e40d11a0' 
	, UUID_SERVICE = '7c5203ca4a19430184b039b2e40d11a1'
	, UUID_CHARACTERISTIC_CMD = '7c5203ca4a19430184b039b2e40d11a2'
	, UUID_CHARACTERISTIC_VOL = '7c5203ca4a19430184b039b2e40d11a3'
	, UUID_CHARACTERISTIC_TRK = '7c5203ca4a19430184b039b2e40d11a4'	
	, UUID_DESCRIPTION = '7c5203ca4a19430184b039b2e40d11a5';	

// Map of bluetooth player commands trough the CMD characteristic
// The strings represents the itunes osascript commands
var command_map = 
	{ 1 : 'to play'
		, 2 : 'to stop'
		, 3 : 'to next track'
		, 4 : 'to previous track'
		, 5 : 'to set mute to (not mute)'
		}
		
// List of commands, when the current track should be broadcasted		
var return_track_on = [1,3,4];
		
var trackNotifyCallback = null
	, volumeNotifyCallback = null;		

var BlenoPrimaryService = bleno.PrimaryService
	, BlenoCharacteristic = bleno.Characteristic
	, BlenoDescriptor = bleno.Descriptor;

/**
 * The TRK characteristic shows the current track
 */
function TrackCharacteristic() {
  TrackCharacteristic.super_.call(this, {
    uuid: UUID_CHARACTERISTIC_TRK,
    properties: ['read', 'notify'],
    descriptors: [
      new BlenoDescriptor({
        uuid: UUID_DESCRIPTION
        , value: 'Track'
      })
    ]
  });
}

util.inherits(TrackCharacteristic, BlenoCharacteristic);

TrackCharacteristic.prototype.onReadRequest = function(offset, callback) {
  if (offset) {
    callback(BlenoCharacteristic.RESULT_ATTR_NOT_LONG, null);
  } else {
	getTrack(function(track) {
		callback(BlenoCharacteristic.RESULT_SUCCESS, new Buffer(track, "utf-8"));    	
	});
  }
};

TrackCharacteristic.prototype.onSubscribe = function(maxValueSize, updateValueCallback) {
	console.log('Trk subscribe');
	trackNotifyCallback = updateValueCallback;
	
	// on subscribe, send the current track to the subscribers
	setTimeout(
		function() {
			getTrack(function(track) {
				console.log("Track: "+track);
				var buffer = new Buffer(track, "utf-8");
				trackNotifyCallback(buffer);		
			});
		}			
		, 2000);
	
};

/**
 * The CMD characteristic controls the player
 * Player commands are represented by an integer
 * See the command map for the list of available commands
 */
function CommandWriteCharacteristic() {
  CommandWriteCharacteristic.super_.call(this, {
    uuid: UUID_CHARACTERISTIC_CMD,
    properties: ['write'],
    descriptors: [
      new BlenoDescriptor({
        uuid: UUID_DESCRIPTION,
        value: 'Command interface'
      })
    ]
  });
}

util.inherits(CommandWriteCharacteristic, BlenoCharacteristic);

CommandWriteCharacteristic.prototype.onWriteRequest = function(data, offset, withoutResponse, callback) {
	if (offset) {
    	callback(this.RESULT_ATTR_NOT_LONG);
	}	
	var value = data.readUInt8(0);
	console.log("Wert: "+value);
	
	command(value);
	
	callback(BlenoCharacteristic.RESULT_SUCCESS, data);
	
};

/**
 * The VOL characteristic shows the current volume or set the volume
 * Valid range: 0 - 100
 */
function VolumeCharacteristic() {
  VolumeCharacteristic.super_.call(this, {
    uuid: UUID_CHARACTERISTIC_VOL,
    properties: ['write', 'read', 'notify'],
    descriptors: [
      new BlenoDescriptor({
        uuid: UUID_DESCRIPTION,
        value: 'Volume'
      })
    ]
  });
}

util.inherits(VolumeCharacteristic, BlenoCharacteristic);

VolumeCharacteristic.prototype.onWriteRequest = function(data, offset, withoutResponse, callback) {
	if (offset) {
    	callback(this.RESULT_ATTR_NOT_LONG);
	}	
	var value = data.readUInt8(0);
	console.log("Wert: "+value);
	
	setVolume(value);
	
	// after a write, send the current volume to the subscribers	
	getVolume(function(vol) {
		var buffer = new Buffer(1);
		buffer.writeUInt8(parseInt(vol), 0);	
		console.log("Volume: "+vol);
		if(volumeNotifyCallback) {
			volumeNotifyCallback(buffer);		
		}
	});	
	
	callback(BlenoCharacteristic.RESULT_SUCCESS, data);
	
};

VolumeCharacteristic.prototype.onReadRequest = function(offset, callback) {
  if (offset) {
    callback(BlenoCharacteristic.RESULT_ATTR_NOT_LONG, null);
  } else {
	getVolume(function(vol) {
		var buffer = new Buffer(1);
		buffer.writeUInt8(parseInt(vol), 0);
		callback(BlenoCharacteristic.RESULT_SUCCESS, buffer);    	
	});
  }
};

VolumeCharacteristic.prototype.onSubscribe = function(maxValueSize, updateValueCallback) {
	console.log('Vol subscribe');
	volumeNotifyCallback = updateValueCallback;

	// on subscribe, send the current volume to the subscribers	
	
	setTimeout(
		function() {
			getVolume(function(vol) {
				console.log("Volume: "+vol);
				var buffer = new Buffer(1);
				buffer.writeUInt8(parseInt(vol), 0);	
				volumeNotifyCallback(buffer);		
			});
		}
		, 2000);
};


function CommandService() {
  CommandService.super_.call(this, {
    uuid: UUID_SERVICE,
    characteristics: [
      new CommandWriteCharacteristic()
      , new VolumeCharacteristic()      
      , new TrackCharacteristic()
    ]
  });
}
util.inherits(CommandService, BlenoPrimaryService);

bleno.on('stateChange', function(state) {
  console.log('on -> stateChange: ' + state);

  if (state === 'poweredOn') {
    bleno.startAdvertising('iTunes', [UUID_DEVICE]);
  } else {
    bleno.stopAdvertising();
  }
});

bleno.on('advertisingStart', function(error) {
  console.log('on -> advertisingStart: ' + (error ? 'error ' + error : 'success'));
  
  if (!error) {
    bleno.setServices([
		new CommandService()
    ]);
  }
});
	
/**
 * Call osascript on bluetooth command 
 */	
function command(cmd) {

	var command = '';
	
	if(command_map.hasOwnProperty(cmd)) {
		command = command_map[cmd]
	}
	
	if('' != command) {
		exec("osascript -e 'tell application \"iTunes\" "+command+"'", function(err, stdout, stderr) {});	
		
		for(var i = 0, l = return_track_on.length; i < l; i++) {
			if(cmd == return_track_on[i]) {
				if(null != trackNotifyCallback) {
					getTrack(function(track) {
						console.log(track);
						trackNotifyCallback(new Buffer(track, 'utf-8'));
					});
				}
				break;
			}
		}
		
	}
	

}	

/**
 * Specific osascript commands are required to get/set the volume and get the track
 */	
function setVolume(vol) {

	exec("osascript -e 'tell application \"iTunes\" to set sound volume to "+vol+"'", function(err, stdout, stderr) {});	

}

function getVolume(done) {

	exec("osascript -e 'tell application \"iTunes\" to get sound volume'", function(err, stdout, stderr) {
		done(stdout);
	});	

}

function getTrack(done)	{
	
	exec("osascript -e 'tell application \"iTunes\" to get the name of the current track'", function(err, stdout, stderr) {
		done(stdout);
	});
	
}